<?php

require_once '../config/database.php';
require_once '../config/cors.php';
require_once '../utils/response.php';
require_once '../utils/auth_middleware.php';

setCORS();

$user_id = AuthMiddleware::verifyToken();
$data = json_decode(file_get_contents('php://input'), true);

if (empty($data['email']) || empty($data['data'])) {
    Response::error('Données manquantes', 'MISSING_DATA', 400);
}

$pdo = Database::connect();

try {
    $pdo->beginTransaction();
    
    // Récupérer les données actuelles du serveur
    $stmt = $pdo->prepare('
        SELECT data_json, last_sync, version FROM user_data WHERE user_id = ?
    ');
    $stmt->execute([$user_id]);
    $remote_data = $stmt->fetch();
    
    $local_data = $data['data'];
    $local_timestamp = strtotime($data['lastSync'] ?? date('c'));
    
    if (!$remote_data) {
        // Première synchronisation
        $new_version = 1;
        $stmt = $pdo->prepare('
            INSERT INTO user_data (user_id, data_json, version) 
            VALUES (?, ?, ?)
        ');
        $stmt->execute([$user_id, json_encode($local_data), $new_version]);
        
        $pdo->commit();
        Response::success([
            'data' => $local_data,
            'version' => $new_version,
            'status' => 'synced'
        ], 'Synchronisation complétée', 200);
    }
    
    $remote_data_obj = json_decode($remote_data['data_json'], true);
    $remote_timestamp = strtotime($remote_data['last_sync']);
    $current_version = $remote_data['version'];
    
    // Logique de fusion basée sur timestamps
    if ($local_timestamp > $remote_timestamp) {
        // Les données locales sont plus récentes
        $merged_data = $local_data;
        $new_version = $current_version + 1;
        $response_status = 'updated_remote';
    } else if ($remote_timestamp > $local_timestamp) {
        // Les données serveur sont plus récentes
        $merged_data = $remote_data_obj;
        $new_version = $current_version;
        $response_status = 'updated_local';
    } else {
        // Timestamps identiques
        $merged_data = $remote_data_obj;
        $new_version = $current_version;
        $response_status = 'no_change';
    }
    
    // Mettre à jour les données serveur
    $stmt = $pdo->prepare('
        UPDATE user_data 
        SET data_json = ?, version = ? 
        WHERE user_id = ?
    ');
    $stmt->execute([json_encode($merged_data), $new_version, $user_id]);
    
    $pdo->commit();
    
    Response::success([
        'data' => $merged_data,
        'version' => $new_version,
        'status' => $response_status,
        'timestamp' => date('c')
    ], 'Synchronisation complétée', 200);
    
} catch (Exception $e) {
    $pdo->rollBack();
    error_log($e->getMessage());
    Response::error('Erreur synchronisation', 'SYNC_ERROR', 500);
}
